//=============================================================================
// NekoGakuen_MulitLanguage.js
// Version: 1.8.1
//=============================================================================
/*:zh
 * @target MZ
 * @plugindesc 多國語言文本 (Ver 1.8.1)
 * @author Mirai
 * @version 1.8.1
 * @url https://nekogakuen.blogspot.com
 * @help
 * 
 * ─ 插件簡介 ─
 * 在RPG Maker MV/MZ中用於支援各國語言文本內容的切換功能。
 * 
 * 
 * ─ 更新履歷 ─
 * V1.8.1 修正MZ在1.3.0版本更新時，無法讀取存放在子資料夾內的插件。 
 * V1.8.0 新增初期語言選擇視窗的自動對齊，並修正初期語言使用背景圖片時的顯示問題。 
 * V1.7.8 修正對話內容使用半形逗號之斷行問題。 
 * V1.7.7 修正初始化語言選擇之後無法按照順序讀取的問題。 
 * V1.7.6 修正選項設定畫面無法即時更新文字選項，以及空白內容選擇的問題。 
 * V1.7.5 修正一些功能和最佳化程式碼，並更新插件的使用條款及移除指定CSV檔案的插件命令。 
 * V1.7.0 新增依語言更改字型功能、指定圖片路徑，以及新增指定CSV檔案的插件命令/文字指令。
 * V1.6.4 更新插件使用條款。
 * V1.6.3 修正在MV/MZ版本時少部分的用語方面顯示文字訊息等問題。
 * V1.6.2 修正在MV/MZ版本時用語方面顯示文字訊息等問題。
 * V1.6.1 修正在MV版本時顯示地圖名稱錯誤的問題。
 * V1.6.0 將MV版本用及MZ版本用的插件合併。
 * V1.5.2 修正遊戲初期選擇語言功能的判斷問題。
 * V1.5.1 將遊戲初期選擇語言程式碼做最佳化處理。
 * V1.5.0 新增遊戲初期選擇語言的功能。
 * V1.4.1 修正「\Say[<參數>]」使用時無法轉換控制字符的問題。
 * V1.4.0 修正「!Say <參數>」使用時的錯誤問題，另外也新增插件使用說明的Google文件連結。
 * V1.3.0 修正「顯示文字..」時名字框無法讀取「外部文本」問題，同時也將程式碼做最佳化處理。
 * V1.2.0 修正讀檔時無法成功讀取「外部文本」參數問題。
 * V1.1.0 修正一些錯誤。
 * V1.0.0 初次版本的插件發佈。
 * 
 * 
 * ─ 使用說明 ─
 * 1.在RPG Maker MV/MZ的「插件管理器」之中載入本插件，
 *   並在本插件的「參數」區塊設定即可。
 * 2.在事件頁中高級區塊選擇「插件命令...」，
 *   並設定選擇要執行的插件命令及參數即可。
 * 
 * 
 * ─ 插件命令/腳本/文字指令 ─
 * 
 * 【變更遊戲語言】
 * --說明：設定文本語言參數，且必須跟CSV檔案裡的欄位名稱一致。
 * --插件命令 MulitLang SET <參數>
 * --腳本 $gameSystem.mulitLangSET("<參數>");
 * 
 * 【呼叫多國語言文本】
 * --說明：在遊戲中呼叫目前所設定的語言文本。
 * 而<參數>為你在CSV檔案裡所輸入的參數名稱。
 * --文字指令  !Say <參數>
 * (※適用於「資料庫」中所有的文字框部分。)
 * --文字指令  !Say <參數1>,<參數2>
 * 而<參數1>為你在CSV檔案裡所輸入的「參數名稱」，<參數2>為你在插件參數「外部文本語言列表」所設定的「參數名稱」。
 * 
 * --文字指令  \Say[<參數>]
 * (※僅在事件指令「顯示文字...」、「顯示選擇...」使用。)
 * --文字指令  \Say[<參數1>,<參數2>]
 * 而<參數1>為你在CSV檔案裡所輸入的「參數名稱」，<參數2>為你在插件參數「外部文本語言列表」所設定的「參數名稱」。
 * 
 * ※注意：上述提到的參數命名方式最好不要有任何的半形空白，
 * 如有空白可以改成「_」取代半形空白。
 * 
 * 
 * ─ 版權聲明 ─
 * 修改或翻譯本插件無需向作者事前告知，但修改後的版本禁止再次發佈。
 * 如果官方的版本有BUG，可以跟作者回報。
 * 
 * 禁止利用本插件進行非法販售及詐騙。
 * 作者只單純提供此插件，如有本插件以外的問題發生請使用者自行負責所有責任。
 * 本插件著作權為貓咪學園(Neko Gakuen)的程式人員Mirai(快閃小強)所有。
 * 
 * --------------------
 * -來源標示：【△ 不需要，但有的話會很感謝。 (註1)】
 * -授權方式：【√ 免費】
 * -商業營利：【√ 允許】
 * -改作許可：【√ 允許】
 * -二次配佈：【× 禁止】
 * -成人用途：【√ 允許】
 * -使用範圍：【※ 僅RPG Maker系列】
 * 
 * ※註1：但如有註明的話，可以註明「NekoGakuen」即可。
 * --------------------
 * 
 * 
 * @command MulitLang SET
 * @text 變更遊戲語言
 * @desc 在遊戲中變更遊戲文本的語言。
 *
 * @arg langTextArgs
 * @type string
 * @default zh_TW
 * @text 文本語言參數
 * @desc 設定文本語言參數，且必須跟CSV檔案裡的欄位名稱一致。
 * 
 * 
 * 
 * @param Custom Fontlist
 * @text 自訂字型清單
 * @desc 將字型檔案放在專案目錄fonts資料夾內，在此參數輸入該字型檔的檔名(不包括副檔名)，用不到此參數就空白即可。
 * 但如果選擇「系統內建字型」時，直接輸入「字型名稱」(例如：微軟正黑體等)即可。
 * @parent Font Group
 * @type struct<fonts>[]
 * @default []
 * 
 * @param Lancsv List
 * @text 外部文本語言...
 * @desc 匯入各種外部文本CSV檔案。
 * @type struct<Lancsv>[]
 * @default ["{\"Lancsv Name\":\"Text01\",\"Lancsv Path\":\"data/Text.csv\"}"]
 * 
 * @param Custom Langlist
 * @text 自訂遊戲語言...
 * @desc 設定自己遊戲可以選擇切換的語言。
 * @type struct<Langlist>[]
 * @default ["{\"Lang Key\":\"zh_TW\",\"Lang Name\":\"中文\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"請選擇語言？\"}","{\"Lang Key\":\"JP\",\"Lang Name\":\"日本語\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"言語を選択してください？\"}","{\"Lang Key\":\"EN\",\"Lang Name\":\"English\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"Please select language?\"}"]
 * 
 * @param Config Lang
 * @text 選項語言名稱
 * @desc 設定遊戲設定「選項」的語言設定名稱，
 * 如需要選項語言支援多國語言可以使用 !Say <參數> 。
 * @type string
 * @default 語言設定
 * 
 * @param LanImage Boolean
 * @text 語言圖片路徑
 * @desc 是否需要開啟在遊戲中指定對應語言圖片的檔案路徑功能。
 * (註：此功能目前不支援手機作業系統。)
 * @type boolean
 * @on 開啟
 * @off 關閉
 * @default false
 * 
 * 
 * @param InitLang Set
 * @text 初期語言選擇...
 * @desc 設定遊戲一開始的語言選擇參數。
 * @type struct<InitLan>
 * @default {"InitLan Switch":"true","InitLan Images":"","InitLan AlignTypeX":"Auto","InitLan Auto SelectX":"Center","InitLan Custom SelectX":"283","InitLan AlignTypeY":"Auto","InitLan Auto SelectY":"Center","InitLan Custom SelectY":"250","InitLan Select Width":"250","InitLan Select Opacity":"255","InitLan Select Align":"left","InitLan HelpX":"0","InitLan HelpY":"0","InitLan Help Opacity":"255"}
 * 
 */
/*~struct~Lancsv:zh
 * 
 * @param Lancsv Name
 * @text 外部文本參數
 * @desc 指定外部文本CSV檔的參數名稱。
 * @type string
 * 
 * @param Lancsv Path
 * @text 外部文本CSV檔
 * @desc 指定外部文本的CSV檔案路徑。
 * @type string
 * 
 */
/*~struct~Langlist:zh
 * 
 * @param Lang Key
 * @text 文本語言參數
 * @desc 指定該遊戲語言的參數名稱，必須跟CSV檔案的欄位名稱一致。
 * @type string
 * @default zh_TW
 * 
 * @param Lang Name
 * @text 文本語言名稱
 * @desc 指定該遊戲語言的顯示名稱。
 * @type string
 * @default 中文
 * 
 * @param Lang FontFace
 * @text 文本語言字型名稱
 * @desc 指定該遊戲語言所顯示的字型名稱，記得先在「自訂字型清單」參數那邊做好設定。
 * @type string
 * @default mplus-1m-regular
 * 
 * @param Lang FontSize
 * @text 文本語言字型大小
 * @desc 指定該遊戲語言所顯示的字型大小。
 * @type number
 * @default 28
 * 
 * @param Init Help
 * @text 初期語言說明
 * @desc 指定初期選擇遊戲語言時顯示的文字說明，
 * 如果「初期語言選擇」為關閉時可不必填此參數。
 * @type string
 * @default 請選擇語言？
 * 
 */
/*~struct~InitLan:zh
 * 
 * @param InitLan Switch
 * @text 初期語言選擇
 * @desc 是否需要開啟遊戲初期時的語言選擇。
 * @type boolean
 * @on 開啟
 * @off 關閉
 * @default true
 * 
 * @param InitLan Images
 * @text 初期語言背景
 * @desc 指定初期語言選擇畫面的背景圖片，
 * 圖片檔案放在img/pictures資料夾內。
 * @type file
 * @dir img/pictures/
 * @require 1
 * 
 * @param InitLan AlignTypeX
 * @text 語言選擇X軸對齊
 * @desc 指定語言選擇視窗的X軸對齊類型。
 * @type select
 * @default Auto
 * @option 自動對齊
 * @value Auto
 * @option 自訂位置
 * @value Custom
 * 
 * @param InitLan Auto SelectX
 * @text 自動對齊X軸座標
 * @desc 指定語言選擇視窗的X軸顯示座標。
 * @type select
 * @parent InitLan AlignTypeX
 * @default Center
 * @option 靠左
 * @value Left
 * @option 置中
 * @value Center
 * @option 靠右
 * @value Right
 * 
 * @param InitLan Custom SelectX
 * @text 自訂位置X軸座標
 * @desc 指定語言選擇視窗的X軸顯示座標。
 * @type number
 * @parent InitLan AlignTypeX
 * @min -9999
 * @default 283
 * 
 * @param InitLan AlignTypeY
 * @text 語言選擇Y軸對齊
 * @desc 指定語言選擇視窗的Y軸對齊類型。
 * @type select
 * @default Auto
 * @option 自動對齊
 * @value Auto
 * @option 自訂位置
 * @value Custom
 * 
 * @param InitLan Auto SelectY
 * @text 自動對齊Y軸座標
 * @desc 指定語言選擇視窗的Y軸顯示座標。
 * @type select
 * @parent InitLan AlignTypeY
 * @default Center
 * @option 靠上
 * @value Top
 * @option 置中
 * @value Center
 * @option 靠下
 * @value Bottom
 * 
 * @param InitLan Custom SelectY
 * @text 自訂位置Y軸座標
 * @desc 指定語言選擇視窗的Y軸顯示座標。
 * @type number
 * @parent InitLan AlignTypeY
 * @min -9999
 * @default 250
 * 
 * @param InitLan Select Width
 * @text 語言選擇視窗寬度
 * @desc 指定語言選擇視窗的顯示寬度。
 * @type number
 * @min 100
 * @default 250
 * 
 * @param InitLan Select Opacity
 * @text 語言選擇不透明度
 * @desc 指定語言選擇視窗的不透明度。
 * @type number
 * @min 0
 * @max 255
 * @default 255
 * 
 * @param InitLan Select Align
 * @text 語言選擇文字對齊
 * @desc 指定語言選擇視窗的文字顯示位置對齊。
 * @type select
 * @default left
 * @option 靠左
 * @value left
 * @option 置中
 * @value center
 * @option 靠右
 * @value right
 * 
 * @param InitLan HelpX
 * @text 語言說明X軸
 * @desc 指定語言說明視窗的X軸顯示座標。
 * @type number
 * @min -9999
 * @default 0
 * 
 * @param InitLan HelpY
 * @text 語言說明Y軸
 * @desc 指定語言說明視窗的Y軸顯示座標。
 * @type number
 * @min -9999
 * @default 0
 * 
 * @param InitLan Help Opacity
 * @text 語言說明不透明度
 * @desc 指定語言說明視窗的不透明度。
 * @type number
 * @min 0
 * @max 255
 * @default 255
 * 
 */
/*~struct~fonts:zh
 * 
 * @param Fonts File
 * @text 指定字型檔案名稱
 * @desc 指定字型檔案名稱，但不包含副檔名，但如果選擇「系統內建字型」，
 * 則會抓取電腦系統本身已經安裝好的字型。
 * @type string
 * @default mplus-1m-regular
 * 
 * @param Fonts Format
 * @text 指定字型格式
 * @desc 指定字型的副檔名格式，但如果選擇「系統內建字型」，
 * 則會抓取電腦系統本身已經安裝好的字型。
 * @type select
 * @default ttf
 * @option 系統內建字型
 * @value local
 * @option OTF (OpenType Font)
 * @value otf
 * @option TTF (TrueType Font)
 * @value ttf
 * @option WOFF (Web Open Font Format)
 * @value woff
 * @option SVG (Scalable Vector Graphics font)
 * @value svg
 * 
 */
//=============================================================================
'use strict';

let NekoGakuen_MulitLanguage = {};
NekoGakuen_MulitLanguage.PluginName = decodeURIComponent(document.currentScript.src).replace(/^.*?\/plugins\/(.*).js$/, "$1");
NekoGakuen_MulitLanguage.Parameters = PluginManager.parameters(NekoGakuen_MulitLanguage.PluginName);
NekoGakuen_MulitLanguage.Config_Lang = String(NekoGakuen_MulitLanguage.Parameters['Config Lang'] || "語言設定");
NekoGakuen_MulitLanguage.LanImage_Boolean = String(NekoGakuen_MulitLanguage.Parameters['LanImage Boolean'] || "false");
NekoGakuen_MulitLanguage.cfl = JSON.parse(NekoGakuen_MulitLanguage.Parameters['Custom Fontlist']);
NekoGakuen_MulitLanguage.fonts_file = [];
NekoGakuen_MulitLanguage.fonts_format = [];

const args_InitLanSet = JSON.parse(NekoGakuen_MulitLanguage.Parameters['InitLang Set']);
const args_LanNameList = JSON.parse(NekoGakuen_MulitLanguage.Parameters['Custom Langlist']);
const args_LancsvFileList = JSON.parse(NekoGakuen_MulitLanguage.Parameters['Lancsv List']);
const args_Lancsv1a = Array();
const args_Lancsv1b = Array();
const args_Lan2a = Array();
const args_Lan2b = Array();
const args_Lan2d = Array();
const args_Lan2e = Array();
const args_Lan2c = Array();
let args_LancsvPath, args_LanName;

if (Utils.RPGMAKER_NAME === "MZ") {
    (() => {
        PluginManager.isPlugins = function (pluginsName) {
            return this._scripts.includes(pluginsName);
        };

        PluginManager.checkFile = function (url) {
            const xmlhttp = new XMLHttpRequest();
            xmlhttp.open("GET", url, false);
            xmlhttp.send(null);
            if (xmlhttp.readyState == 4) {
                if (xmlhttp.status == 200) {
                    return true;
                } else {
                    return false;
                }
            }
        }

        PluginManager.convText = function (text) {
            const args = text.split(" ");
            const command = args.shift();
            if (command === "!Say") {
                const lanargs = args[0].split(",");
                if (lanargs[1]) {
                    return MulitLanguageArgs.getLangDataTextS(lanargs[0], lanargs[1]);
                } else {
                    return MulitLanguageArgs.getLangDataTextO(lanargs[0]);
                }
            } else {
                return text;
            }
        }

        for (let i = 0; i < args_LancsvFileList.length; i++) {
            args_LancsvPath = JSON.parse(args_LancsvFileList[i]);
            args_Lancsv1a.push(String(args_LancsvPath["Lancsv Name"]));
            args_Lancsv1b.push(String(args_LancsvPath["Lancsv Path"]));
        }

        for (let i = 0; i < args_LanNameList.length; i++) {
            args_LanName = JSON.parse(args_LanNameList[i]);
            args_Lan2a.push(String(args_LanName["Lang Key"]));
            args_Lan2b.push(String(args_LanName["Lang Name"]));
            args_Lan2d.push(String(args_LanName["Lang FontFace"]));
            args_Lan2e.push(Number(args_LanName["Lang FontSize"]));
            args_Lan2c.push(String(args_LanName["Init Help"]));
        }
        const fs = Utils.isNwjs() ? require('fs') : null;
        const path = Utils.isNwjs() ? require("path") : null;
        const base = Utils.isNwjs() ? path.dirname(process.mainModule.filename) : null;

        NekoGakuen_MulitLanguage._Scene_Boot_start = Scene_Boot.prototype.start;
        Scene_Boot.prototype.start = function () {
            NekoGakuen_MulitLanguage._Scene_Boot_start.call(this);
            if (Utils.isNwjs()) {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !fs.existsSync(path.join(base, 'save/config.rmmzsave'))) {
                    ConfigManager.language = "init";
                }
            } else {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !localforage.getItem('config')) {
                    ConfigManager.language = "init";
                }
            }
            if (ConfigManager.language == "init") {
                if (!DataManager.isBattleTest() && !DataManager.isEventTest()) {
                    SceneManager.goto(Scene_InitialLanguage);
                }
            }
        };

        let args_Csvindex = args_Lancsv1b[0];
        let args_Lanindex = args_Lan2a[0];
        const request = new XMLHttpRequest();
        request.open("GET", args_Csvindex, false);
        request.send(null);

        const csvData = new Array();
        const jsonObject = request.responseText.split(/\r\n|\r/);
        if (Utils.isNwjs()) {
            if (fs.existsSync(path.join(base, args_Csvindex))) {
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
            }
        } else {
            if (PluginManager.checkFile(args_Csvindex)) {
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
            }
        }

        PluginManager.registerCommand(NekoGakuen_MulitLanguage.PluginName, "MulitLang SET", args => {
            if (args_Lan2a[args_Lan2a.indexOf(args.langTextArgs)] != undefined) {
                $gameSystem.mulitLangSET(args.langTextArgs);
            }
        });

        if (!PluginManager.isPlugins("NekoGakuen_FontManager")) {
            for (let i = 0; i < NekoGakuen_MulitLanguage.cfl.length; i++) {
                let Read_FontManager = JSON.parse(NekoGakuen_MulitLanguage.cfl[i]);
                NekoGakuen_MulitLanguage.fonts_file.push(Read_FontManager["Fonts File"]);
                NekoGakuen_MulitLanguage.fonts_format.push(Read_FontManager["Fonts Format"]);
            }

            Graphics.localFont = function (name) {
                const style = document.createElement('style');
                const head = document.getElementsByTagName('head');
                const rule = '@font-face { font-family: "' + name + '"; src: local("' + name + '"); }';
                style.type = 'text/css';
                head.item(0).appendChild(style);
                style.sheet.insertRule(rule, 0);
                this._createFontLoader(name);
            };

            Scene_Boot.prototype.loadGameFonts = function () {
                const advanced = $dataSystem.advanced;
                FontManager.load("rmmz-mainfont", advanced.mainFontFilename);
                FontManager.load("rmmz-numberfont", advanced.numberFontFilename);
                for (let i = 0; i < NekoGakuen_MulitLanguage.cfl.length; ++i) {
                    let filename = NekoGakuen_MulitLanguage.fonts_file[i].trim();
                    if (NekoGakuen_MulitLanguage.fonts_format[i] != 'local') {
                        const url = './fonts/' + filename + '.' + NekoGakuen_MulitLanguage.fonts_format[i];
                        if (PluginManager.checkFile(url)) {
                            FontManager.load(filename, filename + '.' + NekoGakuen_MulitLanguage.fonts_format[i]);
                        }
                    } else {
                        Graphics.localFont(filename);
                    }
                }
            };
        }

        Game_System.prototype.mainFontFace = function () {
            if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                return args_Lan2d[ConfigManager.language];
            } else {
                return "rmmz-mainfont, " + $dataSystem.advanced.fallbackFonts;
            }
        };

        Game_System.prototype.mainFontSize = function () {
            if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                return args_Lan2e[ConfigManager.language];
            } else {
                return $dataSystem.advanced.fontSize;
            }
        };

        ImageManager.loadBitmap = function (folder, filename) {
            if (filename) {
                if (NekoGakuen_MulitLanguage.LanImage_Boolean == "true") {
                    if (Utils.isNwjs()) {
                        if (fs.existsSync(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    } else {
                        if (PluginManager.checkFile(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    }
                }
                const url = folder + Utils.encodeURI(filename) + ".png";
                return this.loadBitmapFromUrl(url);
            } else {
                return this._emptyBitmap;
            }
        };


        NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
        Window_Base.prototype.convertEscapeCharacters = function (text) {
            let content = NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters.call(this, text);
            content = content.replace(/\x1bSay\[(.*?)\]/gi, (_, p1) => {
                const args = p1.split(",");
                if (args[1]) {
                    return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1]));
                } else {
                    return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextO(args[0]));
                }
            });
            return content;
        };

        Game_Interpreter.prototype.checkTextByData = function (text) {
            text = text.replace(/\\/g, '\x1b');
            text = text.replace(/\x1bSay\[(.*?)\]/gi, (_, p1) => {
                const args = p1.split(",");
                if (args[1]) {
                    return MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                } else {
                    return MulitLanguageArgs.getLangDataTextO(args[0]);
                }
            });
            return text;
        };

        Game_Interpreter.prototype.setupChoices = function (params) {
            const choices = params[0].clone();
            for (let i = 0; i < choices.length; i++) {
                choices[i] = this.checkTextByData(choices[i]);
            }
            const cancelType = params[1] < choices.length ? params[1] : -2;
            const defaultType = params.length > 2 ? params[2] : 0;
            const positionType = params.length > 3 ? params[3] : 2;
            const background = params.length > 4 ? params[4] : 0;
            $gameMessage.setChoices(choices, defaultType, cancelType);
            $gameMessage.setChoiceBackground(background);
            $gameMessage.setChoicePositionType(positionType);
            $gameMessage.setChoiceCallback(n => {
                this._branch[this._indent] = n;
            });
        };

        TextManager.basic = function (basicId) {
            return PluginManager.convText($dataSystem.terms.basic[basicId]) || '';
        };

        TextManager.param = function (paramId) {
            return PluginManager.convText($dataSystem.terms.params[paramId]) || '';
        };

        TextManager.command = function (commandId) {
            return PluginManager.convText($dataSystem.terms.commands[commandId]) || '';
        };

        TextManager.message = function (messageId) {
            return PluginManager.convText($dataSystem.terms.messages[messageId]) || '';
        };

        Object.defineProperty(TextManager, "currencyUnit", {
            get: function () {
                return PluginManager.convText($dataSystem.currencyUnit);
            },
            configurable: true
        });

        Game_Actor.prototype.name = function () {
            return PluginManager.convText(this._name);
        };

        Game_Actor.prototype.nickname = function () {
            return PluginManager.convText(this._nickname);
        };

        Game_Actor.prototype.profile = function () {
            return PluginManager.convText(this._profile);
        };

        Window_Base.prototype.drawItemName = function (item, x, y, width) {
            if (item) {
                const iconY = y + (this.lineHeight() - ImageManager.iconHeight) / 2;
                const textMargin = ImageManager.iconWidth + 4;
                const itemWidth = Math.max(0, width - textMargin);
                this.resetTextColor();
                this.drawIcon(item.iconIndex, x, iconY);
                this.drawText(PluginManager.convText(item.name), x + textMargin, y, itemWidth);
            }
        };

        Window_StatusBase.prototype.drawActorClass = function (actor, x, y, width) {
            width = width || 168;
            this.resetTextColor();
            this.drawText(PluginManager.convText(actor.currentClass().name), x, y, width);
        };

        Window_Help.prototype.refresh = function () {
            const rect = this.baseTextRect();
            this.contents.clear();
            this.drawTextEx(PluginManager.convText(this._text), rect.x, rect.y, rect.width);
        };

        Window_SkillType.prototype.makeCommandList = function () {
            if (this._actor) {
                const skillTypes = this._actor.skillTypes();
                for (const stypeId of skillTypes) {
                    const name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                    this.addCommand(name, "skill", true, stypeId);
                }
            }
        };

        Window_ActorCommand.prototype.addSkillCommands = function () {
            const skillTypes = this._actor.skillTypes();
            for (const stypeId of skillTypes) {
                const name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                this.addCommand(name, "skill", true, stypeId);
            }
        };

        Window_BattleLog.prototype.displayItemMessage = function (fmt, subject, item) {
            if (fmt) {
                this.push("addText", PluginManager.convText(fmt).format(subject.name(), PluginManager.convText(item.name)));
            }
        };

        Window_StatusBase.prototype.actorSlotName = function (actor, index) {
            const slots = actor.equipSlots();
            return PluginManager.convText($dataSystem.equipTypes[slots[index]]);
        };

        Game_Enemy.prototype.battlerName = function () {
            return PluginManager.convText(this.enemy().battlerName);
        };

        Game_Enemy.prototype.originalName = function () {
            return PluginManager.convText(this.enemy().name);
        };

        Game_Map.prototype.displayName = function () {
            return PluginManager.convText($dataMap.displayName);
        };

        Scene_Boot.prototype.updateDocumentTitle = function () {
            document.title = PluginManager.convText($dataSystem.gameTitle);
        };

        NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages = Scene_Boot.prototype.loadSystemImages;
        Scene_Boot.prototype.loadSystemImages = function () {
            NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages.call(this);
            ImageManager.loadPicture(args_InitLanSet["InitLan Images"])
        };

        NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle = Scene_Title.prototype.drawGameTitle;
        Scene_Title.prototype.drawGameTitle = function () {
            $dataSystem.gameTitle = PluginManager.convText($dataSystem.gameTitle);
            if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                this._gameTitleSprite.bitmap.fontFace = args_Lan2d[ConfigManager.language];
            } else {
                this._gameTitleSprite.bitmap.fontFace = $gameSystem.mainFontFace();
            }
            NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle.call(this);
        };


        function Window_LanguageSelect() {
            this.initialize(...arguments);
        }

        Window_LanguageSelect.prototype = Object.create(Window_Selectable.prototype);
        Window_LanguageSelect.prototype.constructor = Window_LanguageSelect;

        Window_LanguageSelect.prototype.initialize = function (rect) {
            Window_Selectable.prototype.initialize.call(this, rect);
            this.opacity = Number(args_InitLanSet["InitLan Select Opacity"] || 255);
            this.width = this.windowWidth();
            this.height = this.windowHeight();
            this.setTopRow(0);
            this.select(0);
            this.activate();
            this.refresh();
        };

        Window_LanguageSelect.prototype.maxCols = function () {
            return 1;
        };

        Window_LanguageSelect.prototype.colSpacing = function () {
            return 16;
        };

        Window_LanguageSelect.prototype.maxItems = function () {
            return args_LanNameList ? args_LanNameList.length : 1;
        };

        Window_LanguageSelect.prototype.windowWidth = function () {
            return Number(args_InitLanSet["InitLan Select Width"] || 250);
        };

        Window_LanguageSelect.prototype.windowHeight = function () {
            return this.fittingHeight(this.numVisibleRows());
        };

        Window_LanguageSelect.prototype.drawItem = function (index) {
            const name = this.itemName(index);
            const rect = this.itemLineRect(index);
            this.resetTextColor();
            this.drawText(name, rect.x, rect.y, rect.width, String(args_InitLanSet["InitLan Select Align"] || "left"));
        };

        Window_LanguageSelect.prototype.numVisibleRows = function () {
            return args_LanNameList.length;
        };

        Window_LanguageSelect.prototype.itemName = function (value) {
            return args_Lan2b[value];
        };

        Window_LanguageSelect.prototype.item = function () {
            return this.itemAt(this.index());
        };

        Window_LanguageSelect.prototype.itemAt = function (index) {
            return args_LanNameList ? this.itemName(index) : null;
        };

        Window_LanguageSelect.prototype.refresh = function () {
            Window_Selectable.prototype.refresh.call(this);
        };


        function Window_LanguageHelp() {
            this.initialize(...arguments);
        }

        Window_LanguageHelp.prototype = Object.create(Window_Base.prototype);
        Window_LanguageHelp.prototype.constructor = Window_LanguageHelp;

        Window_LanguageHelp.prototype.initialize = function (rect) {
            Window_Base.prototype.initialize.call(this, rect);
            this._text = "";
            this.opacity = Number(args_InitLanSet["InitLan Help Opacity"] || 255);
        };

        Window_LanguageHelp.prototype.setText = function (text) {
            if (this._text !== text) {
                this._text = text;
                this.refresh();
            }
        };

        Window_LanguageHelp.prototype.clear = function () {
            this.setText("");
        };

        Window_LanguageHelp.prototype.setItem = function (value) {
            this.setText(args_Lan2c[value]);
        };

        Window_LanguageHelp.prototype.refresh = function () {
            const rect = this.baseTextRect();
            this.contents.clear();
            this.drawTextEx(this._text, rect.x, rect.y, rect.width);
        };


        function Scene_InitialLanguage() {
            this.initialize(...arguments);
        }

        Scene_InitialLanguage.prototype = Object.create(Scene_Base.prototype);
        Scene_InitialLanguage.prototype.constructor = Scene_InitialLanguage;

        Scene_InitialLanguage.prototype.initialize = function () {
            Scene_Base.prototype.initialize.call(this);
        };

        Scene_InitialLanguage.prototype.create = function () {
            Scene_Base.prototype.create.call(this);
            this.createBackground();
            this.createLanguageSelectWindow();
            this.createHelpWindow();
        };

        Scene_InitialLanguage.prototype.createBackground = function () {

            this._background = new Sprite(ImageManager.loadPicture(args_InitLanSet["InitLan Images"]));
            this.addChild(this._background);
            this.scaleSprite(this._background);
            this.centerSprite(this._background);
        };

        Scene_InitialLanguage.prototype.langSelectWindowRect = function () {
            let ww = Number(args_InitLanSet["InitLan Select Width"] || 250);
            let wh = this.calcWindowHeight(args_LanNameList.length, false);
            let wx = 0;
            let wy = 0;


            if (args_InitLanSet["InitLan AlignTypeX"] == "Auto") {
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Left') {
                    wx = 0;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Center') {

                    wx = (Graphics.boxWidth - ww) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Right') {
                    wx = Graphics.boxWidth - ww;
                }


            } else {
                wx = Number(args_InitLanSet["InitLan Custom SelectX"] || 283);
            }


            if (args_InitLanSet["InitLan AlignTypeY"] == "Auto") {


                if (args_InitLanSet["InitLan Auto SelectY"] == 'Top') {
                    wy = 0;
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Center') {
                    wy = (Graphics.boxHeight - wh) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Bottom') {
                    wy = Graphics.boxHeight - wh;
                }

            } else {
                wy = Number(args_InitLanSet["InitLan Custom SelectY"] || 250);
            }


            return new Rectangle(wx, wy, ww, wh);
        };

        Scene_InitialLanguage.prototype.needsCancelButton = function () {
            return false;
        };

        Scene_InitialLanguage.prototype.createLanguageSelectWindow = function () {
            const rect = this.langSelectWindowRect();
            this._langSelectWindow = new Window_LanguageSelect(rect);
            this._langSelectWindow.setHandler("ok", this.onLangSetOk.bind(this));
            this.addChild(this._langSelectWindow);
        };

        Scene_InitialLanguage.prototype.createHelpWindow = function () {
            const rect = this.helpWindowRect();
            this._langhelpWindow = new Window_LanguageHelp(rect);
            this.addChild(this._langhelpWindow);
        };

        Scene_InitialLanguage.prototype.helpWindowRect = function () {
            const wx = Number(args_InitLanSet["InitLan HelpX"]);
            const wy = Number(args_InitLanSet["InitLan HelpY"]);
            const ww = Graphics.boxWidth;
            const wh = this.calcWindowHeight(1, false);
            return new Rectangle(wx, wy, ww, wh);
        };

        Scene_InitialLanguage.prototype.update = function () {
            Scene_Base.prototype.update.call(this);
            this._lastSelect = this._langSelectWindow.index();
            this._langhelpWindow.setItem(this._lastSelect);
        };

        Scene_InitialLanguage.prototype.onLangSetOk = function () {
            let langset = args_Lan2a[this._langSelectWindow.index()]
            ConfigManager.language = this._langSelectWindow.index();
            ConfigManager.save();
            MulitLanguageArgs.setLangData(String(langset));
            this._langSelectWindow.close();
            this.fadeOutAll();
            SceneManager.goto(Scene_Boot);
        };


        Window_Options.prototype.statusText = function (index) {
            const symbol = this.commandSymbol(index);
            const value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                return this.volumeStatusText(value);
            } else if (this.isLanguageSymbol(symbol)) {
                return this.langSymbol(value);
            } else {
                return this.booleanStatusText(value);
            }
        };

        Window_Options.prototype.isLanguageSymbol = function (symbol) {
            return symbol.contains('language');
        };

        Window_Options.prototype.langSymbol = function (value) {
            return args_Lan2b[value];
        }

        NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
        Window_Options.prototype.addGeneralOptions = function () {
            this.addCommand(PluginManager.convText(NekoGakuen_MulitLanguage.Config_Lang), 'language');
            NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions.call(this);
        };

        Window_Options.prototype.processOk = function () {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            let value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                this.changeVolume(symbol, true, true);
            } else if (this.isLanguageSymbol(symbol)) {
                value += this.langOffset();
                let config = args_Lan2a;
                if (value > config.length - 1) {
                    value = 0;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            } else {
                this.changeValue(symbol, !this.getConfigValue(symbol));
            }
        };

        Window_Options.prototype.langOffset = function () {
            return 1;
        }

        Window_Options.prototype.cursorLeft = function () {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            let value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                this.changeVolume(symbol, false, false);
            } else if (this.isLanguageSymbol(symbol)) {
                value--;
                value = value.clamp(0, args_LanNameList.length - 1);
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            } else {
                this.changeValue(symbol, false);
            }
        };

        Window_Options.prototype.cursorRight = function () {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            let value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                this.changeVolume(symbol, true, false);
            } else if (this.isLanguageSymbol(symbol)) {
                value++;
                value = value.clamp(0, args_LanNameList.length - 1);
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            } else {
                this.changeValue(symbol, true);
            }
        };

        NekoGakuen_MulitLanguage._configManager_makeData = ConfigManager.makeData;
        ConfigManager.makeData = function () {
            const config = NekoGakuen_MulitLanguage._configManager_makeData.call(this);
            config.language = this.language;
            return config;
        };

        NekoGakuen_MulitLanguage._configManager_applyData = ConfigManager.applyData;
        ConfigManager.applyData = function (config) {
            NekoGakuen_MulitLanguage._configManager_applyData.call(this, config);
            this.language = this.readVolume(config, 'language');
            args_Lanindex = args_Lan2a[this.language];
        };

        ConfigManager.readVolume = function (config, name) {
            const value = config[name];
            if (name != 'language') {
                if (name in config) {
                    return Number(config[name]).clamp(0, 100);
                } else {
                    return 100;
                }
            } else {
                if (name in config) {
                    return Number(config[name]).clamp(0, args_LanNameList.length - 1);
                } else {
                    return 0;
                }
            }
        };

        Game_System.prototype.mulitLangSET = function (lanargs) {
            ConfigManager.language = args_Lan2a.indexOf(lanargs);
            ConfigManager.save();
            MulitLanguageArgs.setLangData(String(lanargs));
        };

        function MulitLanguageArgs() {
            throw new Error('This is a static class');
        }

        MulitLanguageArgs.setLangData = function (lanindex) {
            if (args_Lan2a[args_Lan2a.indexOf(lanindex)] != undefined) {
                args_Lanindex = args_Lan2a[args_Lan2a.indexOf(lanindex)];
            }
        }

        MulitLanguageArgs.getLangDataTextS = function (textArgs, csvArgs) {
            let args_CsvindexS = args_Lancsv1b[args_Lancsv1a.indexOf(csvArgs)];
            let requestS = new XMLHttpRequest();
            requestS.open("GET", args_CsvindexS, false);
            requestS.send(null);
            let csvDataS = new Array();
            let jsonObject = requestS.responseText.split(/\r\n|\r/);
            if (Utils.isNwjs()) {
                if (fs.existsSync(path.join(base, args_CsvindexS))) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataS.push(jsonObject[i].split(';;;'));
                    }
                }
            } else {
                if (PluginManager.checkFile(args_CsvindexS)) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataS.push(jsonObject[i].split(';;;'));
                    }
                }
            }
            let text = '';
            let idList = csvDataS.map(x => x[0]).indexOf(textArgs);
            if (idList > 0) {
                let nameList = csvDataS["0"].indexOf(args_Lanindex);
                text = csvDataS[idList][nameList];
                text = text.replace(/^\"|\"$/g, '');
                return text;
            } else {
                return text;
            }
        }

        MulitLanguageArgs.getLangDataTextO = function (textArgs) {
            let args_CsvindexO = args_Lancsv1b[0];
            let requestO = new XMLHttpRequest();
            requestO.open("GET", args_CsvindexO, false);
            requestO.send(null);
            let csvDataO = new Array();
            let jsonObject = requestO.responseText.split(/\r\n|\r/);
            if (Utils.isNwjs()) {
                if (fs.existsSync(path.join(base, args_CsvindexO))) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataO.push(jsonObject[i].split(';;;'));
                    }
                }
            } else {
                if (PluginManager.checkFile(args_CsvindexO)) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataO.push(jsonObject[i].split(';;;'));
                    }
                }
            }
            let text = '';
            let idList = csvDataO.map(x => x[0]).indexOf(textArgs);
            if (idList > 0) {
                let nameList = csvDataO["0"].indexOf(args_Lanindex);
                text = csvDataO[idList][nameList];
                text = text.replace(/^\"|\"$/g, '');
                return text;
            } else {
                return text;
            }
        }
    })();
}

if (Utils.RPGMAKER_NAME === "MV") {
    (function () {
        PluginManager.isPlugins = function (pluginsName) {
            return this._scripts.includes(pluginsName);
        };

        PluginManager.checkFile = function (url) {
            var xmlhttp = new XMLHttpRequest();
            xmlhttp.open("GET", url, false);
            xmlhttp.send(null);
            if (xmlhttp.readyState == 4) {
                if (xmlhttp.status == 200) {
                    return true;
                } else {
                    return false;
                }
            }
        }

        PluginManager.convText = function (text) {
            var args = text.split(" ");
            var command = args.shift();
            if (command === "!Say") {
                var lanargs = args[0].split(",");
                if (lanargs[1]) {
                    return MulitLanguageArgs.getLangDataTextS(lanargs[0], lanargs[1]);
                } else {
                    return MulitLanguageArgs.getLangDataTextO(lanargs[0]);
                }
            } else {
                return text;
            }
        }

        for (let i = 0; i < args_LancsvFileList.length; i++) {
            args_LancsvPath = JSON.parse(args_LancsvFileList[i]);
            args_Lancsv1a.push(String(args_LancsvPath["Lancsv Name"]));
            args_Lancsv1b.push(String(args_LancsvPath["Lancsv Path"]));
        }
        for (let i = 0; i < args_LanNameList.length; i++) {
            args_LanName = JSON.parse(args_LanNameList[i]);
            args_Lan2a.push(String(args_LanName["Lang Key"]));
            args_Lan2b.push(String(args_LanName["Lang Name"]));
            args_Lan2d.push(String(args_LanName["Lang FontFace"]));
            args_Lan2e.push(Number(args_LanName["Lang FontSize"]));
            args_Lan2c.push(String(args_LanName["Init Help"]));
        }
        var fs = Utils.isNwjs() ? require('fs') : null;
        var path = Utils.isNwjs() ? require('path') : null;
        var base = Utils.isNwjs() ? path.dirname(process.mainModule.filename) : null;

        NekoGakuen_MulitLanguage._Scene_Boot_start = Scene_Boot.prototype.start;
        Scene_Boot.prototype.start = function () {
            NekoGakuen_MulitLanguage._Scene_Boot_start.call(this);
            if (Utils.isNwjs()) {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !fs.existsSync(path.join(base, 'save/config.rpgsave'))) {
                    ConfigManager.language = "init";
                }
            } else {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !localStorage.getItem('RPG Config')) {
                    ConfigManager.language = "init";
                }
            }
            if (ConfigManager.language == "init") {
                if (!DataManager.isBattleTest() && !DataManager.isEventTest()) {
                    SceneManager.goto(Scene_InitialLanguage);
                }
            }
        };

        let args_Csvindex = args_Lancsv1b[0];
        let args_Lanindex = args_Lan2a[0];
        let request = new XMLHttpRequest();
        request.open("GET", args_Csvindex, false);
        request.send(null);

        let csvData = new Array();
        let jsonObject = request.responseText.split(/\r\n|\r/);
        if (Utils.isNwjs()) {
            if (fs.existsSync(path.join(base, args_Csvindex))) {
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
            }
        } else {
            if (PluginManager.checkFile(args_Csvindex)) {
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
            }
        }

        NekoGakuen_MulitLanguage._Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
        Game_Interpreter.prototype.pluginCommand = function (command, args) {
            NekoGakuen_MulitLanguage._Game_Interpreter_pluginCommand.call(this, command, args);
            if (command === 'MulitLang') {
                switch (args[0]) {
                    case 'SET':
                        $gameSystem.mulitLangSET(args[1]);
                        break;
                }
            }
        };

        if (!PluginManager.isPlugins("NekoGakuen_FontManager")) {
            Graphics.localFont = function (name) {
                var style = document.createElement('style');
                var head = document.getElementsByTagName('head');
                var rule = '@font-face { font-family: "' + name + '"; src: local("' + name + '"); }';
                style.type = 'text/css';
                head.item(0).appendChild(style);
                style.sheet.insertRule(rule, 0);
                this._createFontLoader(name);
            };

            for (var i = 0; i < NekoGakuen_MulitLanguage.cfl.length; i++) {
                var Read_FontManager = JSON.parse(NekoGakuen_MulitLanguage.cfl[i]);
                NekoGakuen_MulitLanguage.fonts_file.push(Read_FontManager["Fonts File"]);
                NekoGakuen_MulitLanguage.fonts_format.push(Read_FontManager["Fonts Format"]);
            }

            for (var i = 0; i < NekoGakuen_MulitLanguage.cfl.length; ++i) {
                var filename = NekoGakuen_MulitLanguage.fonts_file[i].trim();
                if (NekoGakuen_MulitLanguage.cfl.length > 0) {
                    if (NekoGakuen_MulitLanguage.fonts_format != 'local') {
                        Graphics.loadFont(filename, './fonts/' + filename + '.' + NekoGakuen_MulitLanguage.fonts_format[i]);
                    } else {
                        Graphics.localFont(filename);
                    }
                }
            }
        }

        NekoGakuen_MulitLanguage._Window_Base_standardFontFace = Window_Base.prototype.standardFontFace;
        Window_Base.prototype.standardFontFace = function () {
            NekoGakuen_MulitLanguage._Window_Base_standardFontFace.call(this);
            if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                return args_Lan2d[ConfigManager.language];
            } else {
                return 'GameFont'
            }
        };

        NekoGakuen_MulitLanguage._Window_Base_standardFontSize = Window_Base.prototype.standardFontSize;
        Window_Base.prototype.standardFontSize = function () {
            NekoGakuen_MulitLanguage._Window_Base_standardFontSize.call(this);
            if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                return args_Lan2e[ConfigManager.language];
            } else {
                return 28
            }
        };

        NekoGakuen_MulitLanguage._Window_Base_resetFontSettings = Window_Base.prototype.resetFontSettings;
        Window_Base.prototype.resetFontSettings = function () {
            NekoGakuen_MulitLanguage._Window_Base_resetFontSettings.call(this);
            this.contents.fontFace = this.standardFontFace();
            this.contents.fontSize = this.standardFontSize();
            this.resetTextColor();
        };

        ImageManager.loadBitmap = function (folder, filename, hue, smooth) {
            if (filename) {
                if (NekoGakuen_MulitLanguage.LanImage_Boolean == "true") {
                    if (Utils.isNwjs()) {
                        if (fs.existsSync(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    } else {
                        if (PluginManager.checkFile(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    }
                }
                var path = folder + encodeURIComponent(filename) + '.png';
                var bitmap = this.loadNormalBitmap(path, hue || 0);
                bitmap.smooth = smooth;
                return bitmap;
            } else {
                return this.loadEmptyBitmap();
            }
        };

        ImageManager.reserveBitmap = function (folder, filename, hue, smooth, reservationId) {
            if (filename) {
                if (NekoGakuen_MulitLanguage.LanImage_Boolean == "true") {
                    if (Utils.isNwjs()) {
                        if (fs.existsSync(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    } else {
                        if (PluginManager.checkFile(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    }
                }
                var path = folder + encodeURIComponent(filename) + '.png';
                var bitmap = this.reserveNormalBitmap(path, hue || 0, reservationId || this._defaultReservationId);
                bitmap.smooth = smooth;
                return bitmap;
            } else {
                return this.loadEmptyBitmap();
            }
        };

        ImageManager.requestBitmap = function (folder, filename, hue, smooth) {
            if (filename) {
                if (NekoGakuen_MulitLanguage.LanImage_Boolean == "true") {
                    if (Utils.isNwjs()) {
                        if (fs.existsSync(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    } else {
                        if (PluginManager.checkFile(folder + args_Lan2a[ConfigManager.language] + '/' + encodeURIComponent(filename) + '.png')) {
                            folder = folder + args_Lan2a[ConfigManager.language] + '/'
                        }
                    }
                }
                var path = folder + encodeURIComponent(filename) + '.png';
                var bitmap = this.requestNormalBitmap(path, hue || 0);
                bitmap.smooth = smooth;
                return bitmap;
            } else {
                return this.loadEmptyBitmap();
            }
        };

        NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
        Window_Base.prototype.convertEscapeCharacters = function (text) {
            let content = NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters.call(this, text);
            content = content.replace(/\x1bSay\[(.*?)\]/gi, function () {
                var args = arguments[1].split(",");
                if (args[1]) {
                    console.log(this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1])));
                    return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1]));
                } else {
                    return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextO(args[0]));
                }
            }.bind(this));
            return content;
        };

        Game_Interpreter.prototype.checkTextByData = function (text) {
            text = text.replace(/\\/g, '\x1b');
            text = text.replace(/\x1bSay\[(.*?)\]/gi, function () {
                var args = arguments[1].split(",");
                if (args[1]) {
                    return MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                } else {
                    return MulitLanguageArgs.getLangDataTextO(args[0]);
                }
            }.bind(this));
            return text;
        };

        Game_Interpreter.prototype.setupChoices = function (params) {
            var choices = params[0].clone();
            for (let i = 0; i < choices.length; i++) {
                choices[i] = this.checkTextByData(choices[i]);
            }
            var cancelType = params[1];
            var defaultType = params.length > 2 ? params[2] : 0;
            var positionType = params.length > 3 ? params[3] : 2;
            var background = params.length > 4 ? params[4] : 0;
            if (cancelType >= choices.length) {
                cancelType = -2;
            }
            $gameMessage.setChoices(choices, defaultType, cancelType);
            $gameMessage.setChoiceBackground(background);
            $gameMessage.setChoicePositionType(positionType);
            $gameMessage.setChoiceCallback(function (n) {
                this._branch[this._indent] = n;
            }.bind(this));
        };

        TextManager.basic = function (basicId) {
            return PluginManager.convText($dataSystem.terms.basic[basicId]) || '';
        };

        TextManager.param = function (paramId) {
            return PluginManager.convText($dataSystem.terms.params[paramId]) || '';
        };

        TextManager.command = function (commandId) {
            return PluginManager.convText($dataSystem.terms.commands[commandId]) || '';
        };

        TextManager.message = function (messageId) {
            return PluginManager.convText($dataSystem.terms.messages[messageId]) || '';
        };

        Object.defineProperty(TextManager, 'currencyUnit', {
            get: function () {
                return PluginManager.convText($dataSystem.currencyUnit);
            },
            configurable: true
        });

        Game_Actor.prototype.name = function () {
            return PluginManager.convText(this._name);
        };

        Game_Actor.prototype.nickname = function () {
            return PluginManager.convText(this._nickname);
        };

        Game_Actor.prototype.profile = function () {
            return PluginManager.convText(this._profile);
        };

        Window_Base.prototype.drawItemName = function (item, x, y, width) {
            width = width || 312;
            if (item) {
                var iconBoxWidth = Window_Base._iconWidth + 4;
                this.resetTextColor();
                this.drawIcon(item.iconIndex, x + 2, y + 2);
                this.drawText(PluginManager.convText(item.name), x + iconBoxWidth, y, width - iconBoxWidth);
            }
        };

        Window_Base.prototype.drawActorClass = function (actor, x, y, width) {
            width = width || 168;
            this.resetTextColor();
            this.drawText(PluginManager.convText(actor.currentClass().name), x, y, width);
        };

        Window_Help.prototype.refresh = function () {
            this.contents.clear();
            this.drawTextEx(PluginManager.convText(this._text), this.textPadding(), 0);
        };

        Window_SkillType.prototype.makeCommandList = function () {
            if (this._actor) {
                var skillTypes = this._actor.addedSkillTypes();
                skillTypes.sort(function (a, b) {
                    return a - b;
                });
                skillTypes.forEach(function (stypeId) {
                    var name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                    this.addCommand(name, 'skill', true, stypeId);
                }, this);
            }
        };

        Window_ActorCommand.prototype.addSkillCommands = function () {
            var skillTypes = this._actor.addedSkillTypes();
            skillTypes.sort(function (a, b) {
                return a - b;
            });
            skillTypes.forEach(function (stypeId) {
                var name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                this.addCommand(name, 'skill', true, stypeId);
            }, this);
        };

        Window_BattleLog.prototype.displayAction = function (subject, item) {
            var numMethods = this._methods.length;
            if (DataManager.isSkill(item)) {
                if (item.message1) {
                    this.push('addText', subject.name() + PluginManager.convText(item.message1).format(PluginManager.convText(item.name)));
                }
                if (item.message2) {
                    this.push('addText', PluginManager.convText(item.message2).format(PluginManager.convText(item.name)));
                }
            } else {
                this.push('addText', PluginManager.convText(TextManager.useItem).format(subject.name(), PluginManager.convText(item.name)));
            }
            if (this._methods.length === numMethods) {
                this.push('wait');
            }
        };

        Window_EquipSlot.prototype.slotName = function (index) {
            var slots = this._actor.equipSlots();
            return this._actor ? PluginManager.convText($dataSystem.equipTypes[slots[index]]) : '';
        };

        Game_Enemy.prototype.battlerName = function () {
            return PluginManager.convText(this.enemy().battlerName);
        };

        Game_Enemy.prototype.originalName = function () {
            return PluginManager.convText(this.enemy().name);
        };

        Game_Map.prototype.displayName = function () {
            return PluginManager.convText($dataMap.displayName);
        };

        Scene_Boot.prototype.updateDocumentTitle = function () {
            document.title = PluginManager.convText($dataSystem.gameTitle);
        };

        NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages = Scene_Boot.loadSystemImages;
        Scene_Boot.loadSystemImages = function () {
            NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages.call(this);
            ImageManager.loadPicture(args_InitLanSet["InitLan Images"]);
        };

        NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle = Scene_Title.prototype.drawGameTitle;
        Scene_Title.prototype.drawGameTitle = function () {
            $dataSystem.gameTitle = PluginManager.convText($dataSystem.gameTitle);
            if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                this._gameTitleSprite.bitmap.fontFace = args_Lan2d[ConfigManager.language];
            } else {
                this._gameTitleSprite.bitmap.fontFace = 'GameFont';
            }
            NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle.call(this);
        };


        function Window_LanguageSelect() {
            this.initialize.apply(this, arguments);
        }

        Window_LanguageSelect.prototype = Object.create(Window_Selectable.prototype);
        Window_LanguageSelect.prototype.constructor = Window_LanguageSelect;

        Window_LanguageSelect.prototype.initialize = function (x, y, width, height) {
            var width = this.windowWidth();
            var height = this.windowHeight();
            Window_Selectable.prototype.initialize.call(this, x, y, width, height);
            this.opacity = Number(args_InitLanSet["InitLan Select Opacity"] || 255);
            this.setTopRow(0);
            this.select(0);
            this.activate();
            this.refresh();
        };

        Window_LanguageSelect.prototype.maxCols = function () {
            return 1;
        };

        Window_LanguageSelect.prototype.spacing = function () {
            return 48;
        };

        Window_LanguageSelect.prototype.maxItems = function () {
            return args_LanNameList ? args_LanNameList.length : 1;
        };

        Window_LanguageSelect.prototype.windowWidth = function () {
            return Number(args_InitLanSet["InitLan Select Width"] || 250);
        };

        Window_LanguageSelect.prototype.windowHeight = function () {
            return this.fittingHeight(this.numVisibleRows());
        };

        Window_LanguageSelect.prototype.drawItem = function (index) {
            var name = args_Lan2b[index];
            var rect = this.itemRectForText(index);
            this.drawText(name, rect.x, rect.y, rect.width, String(args_InitLanSet["InitLan Select Align"] || "left"));
        };

        Window_LanguageSelect.prototype.numVisibleRows = function () {
            return args_LanNameList.length;
        };

        Window_LanguageSelect.prototype.item = function () {
            return args_LanNameList ? args_Lan2b[this.index()] : null;
        };
        Window_LanguageSelect.prototype.makeItemList = function () {

        };
        Window_LanguageSelect.prototype.refresh = function () {
            this.createContents();
            this.drawAllItems();
        };


        function Window_LanguageHelp() {
            this.initialize.apply(this, arguments);
        }

        Window_LanguageHelp.prototype = Object.create(Window_Base.prototype);
        Window_LanguageHelp.prototype.constructor = Window_LanguageHelp;

        Window_LanguageHelp.prototype.initialize = function (numLines) {
            var width = Graphics.boxWidth;
            var height = this.fittingHeight(1);
            Window_Base.prototype.initialize.call(this, 0, 0, width, height);
            this._text = '';
            this.opacity = Number(args_InitLanSet["InitLan Help Opacity"] || 255);
        };

        Window_LanguageHelp.prototype.setText = function (text) {
            if (this._text !== text) {
                this._text = text;
                this.refresh();
            }
        };

        Window_LanguageHelp.prototype.clear = function () {
            this.setText("");
        };

        Window_LanguageHelp.prototype.setItem = function (value) {
            this.setText(args_Lan2c[value]);
        };

        Window_LanguageHelp.prototype.refresh = function () {
            this.contents.clear();
            this.drawTextEx(this._text, this.textPadding(), 0);
        };


        function Scene_InitialLanguage() {
            this.initialize.apply(this, arguments);
        }

        Scene_InitialLanguage.prototype = Object.create(Scene_Base.prototype);
        Scene_InitialLanguage.prototype.constructor = Scene_InitialLanguage;

        Scene_InitialLanguage.prototype.initialize = function () {
            Scene_Base.prototype.initialize.call(this);
        };

        Scene_InitialLanguage.prototype.create = function () {
            Scene_Base.prototype.create.call(this);
            this.createBackground();
            this.createLanguageSelectWindow();
            this.createHelpWindow();
        };

        Scene_InitialLanguage.prototype.createBackground = function () {
            this._background = new Sprite(ImageManager.loadPicture(args_InitLanSet["InitLan Images"]));
            this.addChild(this._background);
            this.scaleSprite(this._background);
            this.centerSprite(this._background);

        };

        Scene_InitialLanguage.prototype.scaleSprite = function (sprite) {
            var scaleX = Graphics.width / sprite.bitmap.baseTexture.width;
            var scaleY = Graphics.height / sprite.bitmap.baseTexture.height;
            var scale = Math.max(scaleX, scaleY, 1.0);
            sprite.scale.x = scale;
            sprite.scale.y = scale;
        };

        Scene_InitialLanguage.prototype.centerSprite = function (sprite) {
            sprite.x = Graphics.width / 2;
            sprite.y = Graphics.height / 2;
            sprite.anchor.x = 0.5;
            sprite.anchor.y = 0.5;
        };

        Scene_InitialLanguage.prototype.createLanguageSelectWindow = function () {
            this._langSelectWindow = new Window_LanguageSelect();
            if (args_InitLanSet["InitLan AlignTypeX"] == "Auto") {
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Left') {
                    this._langSelectWindow.x = 0;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Center') {

                    this._langSelectWindow.x = (Graphics.boxWidth - this._langSelectWindow.width) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Right') {
                    this._langSelectWindow.x = Graphics.boxWidth - this._langSelectWindow.width;
                }


            } else {
                this._langSelectWindow.x = Number(args_InitLanSet["InitLan Custom SelectX"] || 283);
            }


            if (args_InitLanSet["InitLan AlignTypeY"] == "Auto") {


                if (args_InitLanSet["InitLan Auto SelectY"] == 'Top') {
                    this._langSelectWindow.y = 0;
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Center') {
                    this._langSelectWindow.y = (Graphics.boxHeight - this._langSelectWindow.height) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Bottom') {
                    this._langSelectWindow.y = Graphics.boxHeight - this._langSelectWindow.height;
                }

            } else {
                this._langSelectWindow.y = Number(args_InitLanSet["InitLan Custom SelectY"] || 250);
            }

            this._langSelectWindow.setHandler("ok", this.onLangSetOk.bind(this));
            this.addChild(this._langSelectWindow);
        };

        Scene_InitialLanguage.prototype.createHelpWindow = function () {
            this._langhelpWindow = new Window_LanguageHelp();
            this._langhelpWindow.x = Number(args_InitLanSet["InitLan HelpX"]);
            this._langhelpWindow.y = Number(args_InitLanSet["InitLan HelpY"]);
            this.addChild(this._langhelpWindow);
        };

        Scene_InitialLanguage.prototype.update = function () {
            Scene_Base.prototype.update.call(this);
            this._lastSelect = this._langSelectWindow.index();
            this._langhelpWindow.setItem(this._lastSelect);
        };

        Scene_InitialLanguage.prototype.onLangSetOk = function () {
            let langset = args_Lan2a[this._langSelectWindow.index()]
            ConfigManager.language = this._langSelectWindow.index();
            ConfigManager.save();
            MulitLanguageArgs.setLangData(langset);
            this._langSelectWindow.close();
            this.fadeOutAll();
            SceneManager.goto(Scene_Boot);
        };


        Window_Options.prototype.statusText = function (index) {
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                return this.volumeStatusText(value);
            } else if (this.isLanguageSymbol(symbol)) {

                return this.langSymbol(value);
            } else {
                return this.booleanStatusText(value);
            }
        };

        Window_Options.prototype.isLanguageSymbol = function (symbol) {
            return symbol.contains('language');
        };

        Window_Options.prototype.langSymbol = function (value) {
            return args_Lan2b[value];
        }

        NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
        Window_Options.prototype.addGeneralOptions = function () {
            this.addCommand(PluginManager.convText(NekoGakuen_MulitLanguage.Config_Lang), 'language');
            NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions.call(this);
        };

        Window_Options.prototype.processOk = function () {
            var index = this.index();
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                value += this.volumeOffset();
                if (value > 100) {
                    value = 0;
                }
                value = value.clamp(0, 100);
                this.changeValue(symbol, value);
            } else if (this.isLanguageSymbol(symbol)) {
                value += this.langOffset();
                let config = args_Lan2a;
                if (value > config.length - 1) {
                    value = 0;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            } else {
                this.changeValue(symbol, !value);
            }
        };

        Window_Options.prototype.langOffset = function () {
            return 1;
        }

        Window_Options.prototype.cursorLeft = function (wrap) {
            var index = this.index();
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                value -= this.volumeOffset();
                value = value.clamp(0, 100);
                this.changeValue(symbol, value);
            } else if (this.isLanguageSymbol(symbol)) {
                value--;
                value = value.clamp(0, args_LanNameList.length - 1);
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            } else {
                this.changeValue(symbol, false);
            }
        };

        Window_Options.prototype.cursorRight = function (wrap) {
            var index = this.index();
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                value += this.volumeOffset();
                value = value.clamp(0, 100);
                this.changeValue(symbol, value);
            } else if (this.isLanguageSymbol(symbol)) {
                value++;
                value = value.clamp(0, args_LanNameList.length - 1);
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            } else {
                this.changeValue(symbol, true);
            }
        };

        NekoGakuen_MulitLanguage._configManager_makeData = ConfigManager.makeData;
        ConfigManager.makeData = function () {
            var config = NekoGakuen_MulitLanguage._configManager_makeData.call(this);
            config.language = this.language;
            return config;
        };

        NekoGakuen_MulitLanguage._configManager_applyData = ConfigManager.applyData;
        ConfigManager.applyData = function (config) {
            NekoGakuen_MulitLanguage._configManager_applyData.call(this, config);
            this.language = this.readVolume(config, 'language');
            args_Lanindex = args_Lan2a[this.language];
        };

        ConfigManager.readVolume = function (config, name) {
            var value = config[name];
            if (name != 'language') {
                if (name in config) {
                    return Number(config[name]).clamp(0, 100);
                } else {
                    return 100;
                }
            } else {
                if (name in config) {
                    return Number(config[name]).clamp(0, args_LanNameList.length - 1);
                } else {
                    return 0;
                }
            }
        };

        Game_System.prototype.mulitLangSET = function (lanargs) {
            ConfigManager.language = args_Lan2a.indexOf(lanargs);
            ConfigManager.save();
            MulitLanguageArgs.setLangData(lanargs);
        };

        function MulitLanguageArgs() {
            throw new Error('This is a static class');
        }

        MulitLanguageArgs.setLangData = function (lanindex) {
            if (args_Lan2a[args_Lan2a.indexOf(lanindex)] != undefined) {
                args_Lanindex = args_Lan2a[args_Lan2a.indexOf(lanindex)];
            }
        }

        MulitLanguageArgs.getLangDataTextS = function (textArgs, csvArgs) {
            let args_CsvindexS = args_Lancsv1b[args_Lancsv1a.indexOf(csvArgs)];
            let requestS = new XMLHttpRequest();
            requestS.open("GET", args_CsvindexS, false);
            requestS.send(null);
            let csvDataS = new Array();
            let jsonObject = requestS.responseText.split(/\r\n|\r/);
            if (Utils.isNwjs()) {
                if (fs.existsSync(path.join(base, args_CsvindexS))) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataS.push(jsonObject[i].split(';;;'));
                    }
                }
            } else {
                if (PluginManager.checkFile(args_CsvindexS)) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataS.push(jsonObject[i].split(';;;'));
                    }
                }
            }
            let text = '';
            var idList = csvDataS.map(x => x[0]).indexOf(textArgs);
            if (idList > 0) {
                var nameList = csvDataS["0"].indexOf(args_Lanindex);
                text = csvDataS[idList][nameList];
                text = text.replace(/^\"|\"$/g, '');
                console.log(text);
                text = text.replace(/\\/g, '\x1b');
                console.log(text);
                return text;
            } else {
                return text;
            }
        }

        MulitLanguageArgs.getLangDataTextO = function (textArgs) {
            let args_CsvindexO = args_Lancsv1b[0];
            let requestO = new XMLHttpRequest();
            requestO.open("GET", args_CsvindexO, false);
            requestO.send(null);
            let csvDataO = new Array();
            let jsonObject = requestO.responseText.split(/\r\n|\r/);
            if (Utils.isNwjs()) {
                if (fs.existsSync(path.join(base, args_CsvindexO))) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataO.push(jsonObject[i].split(';;;'));
                    }
                }
            } else {
                if (PluginManager.checkFile(args_CsvindexO)) {
                    for (let i = 0; i < jsonObject.length; i++) {
                        jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                        csvDataO.push(jsonObject[i].split(';;;'));
                    }

                }
            }
            let text = '';
            var idList = csvDataO.map(x => x[0]).indexOf(textArgs);
            if (idList > 0) {
                var nameList = csvDataO["0"].indexOf(args_Lanindex);
                text = csvDataO[idList][nameList];
                text = text.replace(/^\"|\"$/g, '');
                return text;
            } else {
                return text;
            }
        }
    })();
}